// Routines and Data related to settings
#include "settings.h"
#include "flash.h"
#include "io.h"

void defaultSettings(){                     //load sensible defaults
    newUartMode=UART_MODE_MSMOUSE;
//    newUartMode=UART_MODE_MSMOUSE_EXT;  //extended to 8 bits for more depth
//    newUartMode=UART_MODE_ASCII;         //text
    newBaudRate=1200;
    newDPIdivider=1;                      //scaling factor
    newUpdateInterval=100;                //update interval
}

void saveSettings(){
    //put in RAM image
    RAMImage[0]=newUartMode;
    RAMImage[1]=newBaudRate;
    RAMImage[2]=newDPIdivider;
    RAMImage[3]=newUpdateInterval;
    //save RAM image to flash
    if(saveToFlash()){              //non-zero on error
        uart_print("Save failed.\r\n");        
    }else{
        uart_print("Saved OK.\r\n");        
    }
    
}

void loadSettings(){
    //load from flash
    loadToRAM();
    //store in working variables
    newUartMode=RAMImage[0];
    newBaudRate=RAMImage[1];
    newDPIdivider=RAMImage[2];
    newUpdateInterval=RAMImage[3];    
}

void doSetupMenu(){         //main menu setup routine
    int setupDone=0, menuState=0,d;
    uint32_t n=0;                   //for numeric entry
    newBaudRate=currentBaudRate;
    newUartMode=currentUartMode;
    newDPIdivider=currentDPIdivider;
    newUpdateInterval=currentUpdateInterval;
    AMBER_LED=1;
    while(!setupDone){
        SYS_Tasks();                       //Harmony code- USB servicing etc
        if(menuState==0){printMenu(newUartMode,newBaudRate,newDPIdivider,newUpdateInterval);menuState=1;}
        if(uart_available()){
            d=uart_receive();
            if(menuState==1){
                switch(d){
                    case 'X':
                    case 'x':setupDone=1;break;
                    case 'C':
                    case 'c':uart_print("Settings NOT changed. Exiting setting mode.\r\n");AMBER_LED=0;return;break;
                    case 'D':
                    case 'd':uart_print("Loaded defaults.\r\n");defaultSettings();menuState=0;break;
                    case 'L':
                    case 'l':uart_print("Loaded from FLASH.\r\n");loadSettings();menuState=0;break;
                    case 'S':
                    case 's':uart_print("Saving to FLASH.\r\n");saveSettings();menuState=0;break;
                    case '0':uart_print("Baud rate set to 1200.\r\n");newBaudRate=1200;menuState=0;break;
                    case '1':uart_print("Baud rate set to 9600.\r\n");newBaudRate=9600;menuState=0;break;
                    case '2':uart_print("Baud rate set to 115200.\r\n");newBaudRate=115200;menuState=0;break;
                    case '3':uart_print("Enter new Baud rate:");n=0;menuState=2;break;
                    case '4':uart_print("Mode set to 7bit MS Mouse\r\n");newUartMode=UART_MODE_MSMOUSE;menuState=0;break;
                    case '5':uart_print("Mode set to 8bit Extended MS Mouse\r\n");newUartMode=UART_MODE_MSMOUSE_EXT;menuState=0;break;
                    case '6':uart_print("Mode set to CSV text\r\n");newUartMode=UART_MODE_ASCII;menuState=0;break;
                    case '7':uart_print("Enter new DPI Divisor:");n=0;menuState=3;break;
                    case '8':uart_print("Enter new Update Interval:");n=0;menuState=4;break;
//                    case '9':break;
                }
            }else if(menuState>=2){
                if((d>='0')&&(d<='9')){
                    n=n*10+d-'0';
                    uart_send(d);           //echo
                }
                if(((d==13)||(d==10))&&(n)){                  //avoid following CR/LF on Arduino Serial monitor etc
                    uart_print("\r\nEntered:");
                    uart_printl(n);
                    uart_print("\r\n");
                    if(menuState==2){                   //baudrate
                        if((n<100)||(n>1000000L)){      //out of range
                            uart_print("Out of range- not accepted.\r\n");
                        }else{                                              //OK
                            uart_print("Accepted.\r\n");
                            newBaudRate=n;
                        }
                    }
                    if(menuState==3){                 //DPI divisor
                        if((n<1)||(n>1000L)){      //out of range
                            uart_print("Out of range- not accepted.\r\n");
                        }else{                                              //OK
                            uart_print("Accepted.\r\n");
                            newDPIdivider=n;
                        }
                    }
                    if(menuState==4){                   //update interval
                        if((n<1)||(n>1000L)){      //out of range
                            uart_print("Out of range- not accepted.\r\n");
                        }else{                                              //OK
                            uart_print("Accepted.\r\n");
                            newUpdateInterval=n;
                        }
                    }
                    menuState=0;                                        //back up to main menu
                }
            }
        }
    }
    uart_print("Settings changed. Exiting setting mode.\r\n\r\n\r\n");
    if(currentBaudRate!=newBaudRate){     //change baud rate if necessary
        while(U1STAbits.TRMT==0){}          //wait til pending data sent at old baud rate        
        uart_init(newBaudRate);
    }
    currentBaudRate=newBaudRate;
    currentUartMode=newUartMode;
    currentDPIdivider=newDPIdivider;
    currentUpdateInterval=newUpdateInterval;
    AMBER_LED=0;
}

void printMenu(uartMode m, uint32_t b,uint32_t d,uint32_t i){
    uart_print("\r\n\r\nUSB Mouse Serial Interface\r\n");
    uart_print("--------------------------\r\n");
    uart_print("0. Set Baud rate 1200\r\n");
    uart_print("1. Set Baud rate 9600\r\n");
    uart_print("2. Set Baud rate 115200\r\n");
    uart_print("3. Set custom Baud rate\r\n");
    uart_print("4. 7bit MS Mouse\r\n");
    uart_print("5. 8bit Extended MS Mouse\r\n");
    uart_print("6. CSV text\r\n");
    uart_print("7. Set DPI Divisor\r\n");
    uart_print("8. Set Update Interval\r\n");
    //uart_print("9. \r\n");
    uart_print("D. Load defaults (1200/7bit/1/100ms)\r\n");
    uart_print("L. Load from FLASH\r\n");
    uart_print("S. Save to FLASH\r\n");
    uart_print("X. Exit (with new settings)\r\n");
    uart_print("C. Cancel (don't change settings)\r\n");    
    uart_print("New UART mode: ");
    switch(m){
        case UART_MODE_MSMOUSE: uart_print("7bit MS Mouse\r\n");break;
        case UART_MODE_MSMOUSE_EXT: uart_print("8bit Extended MS Mouse\r\n");break;
        case UART_MODE_ASCII: uart_print("CSV text\r\n");break;
        default: uart_print("Error- not set\r\n");break;
    }
    uart_print("New UART baud rate: ");   
    uart_printl(b);
    uart_print(" baud\r\n");
    uart_print("New DPI Divider: ");   
    uart_printl(d);
    uart_print("\r\n");
    uart_print("New Update Interval: ");   
    uart_printl(i);
    uart_print(" ms\r\n");
}

